pro Timebox, tstart, tend, day=day, nomins=nomins, zero=zero, $
             exact=exact, ytype=ytype, yrange=yrange, yexact=yexact, $
             charsize=charsize, title=title, xtitle=xtitle, $
             ytitle=ytitle, nolabels=nolabels, minutes=minutes, $
             noerase=noerase, date=date

;+
;  TIMEBOX
;	Create axis box with time labelling.
;	This is sort of equivalent to a plot, /nodata, /noerase call.
;
;  Usage:
;	timebox, tstart, tend, [/day, /nomins, /zero, exact=exact, $
;		ytype=ytype, yrange=yrange, yexact=yexact, charsize=charsize, $
;		title = title, xtitle=xtitle, ytitle=ytitle, /nolabels, $
;		minutes=minutes, /noerase, /date]
;
;  Arguments:
;	tstart	float	input	The start time in hours.
;	tend	float	input	The end time in hours.
;
;  Keywords:
;	day	int	input	Day of start - 
;				2-elements (year, day of year). 
;				Required if > 2-months of data
;				are to be plotted.
;	nomins	undef	input	Supress minutes when tick mark spacing is
;				1 hour < spacing < 1 day.
;	zero	undef	input	Supress printing of day at hours other than
;				zero.
;	exact	undef	input	If set then the time range is exact.
;
;    *	ytype	undef	input	If set logarithmic Y - axis.
;    *	yrange	float	input	2-element range of y values.
;    *	yexact	int	input	If set use exact y ranges (actually passed
;				directly to PLOT as YSTYLE.
;
;    *	charsize float	input	Character size to use.
;    *	title	string	input	Plot title.
;	xtitle	string	input	X-axis title.
;    *	ytitle	string	input	Y-axis title.
;	nolabels ?	input	If set then don't plot any labels on the plot.
;	minutes	int	output	Set to 1 if time range < 2 mins else 0.
;       noerase  ?      input   If explicitly zero, then do an erase
;				if unset or non-zero, then don't erase
;	date	?	input	If set, then use month & date rather
;				than day of year.
;
;		N.B. Those keys indicated with an asterisk are passed directly
;			to the PLOT procedure.
;
;  Restrictions:
;	Not all the keys of PLOT are supported - but many of these can be 
;	emulated by setting things in !P, !X and !Y.
;	For time ranges > 4 months the time range is always exact.
;
;  Note:
;	This routine has been written to replace the TIMETICKS routine which
;	although it produced nice results most of the time could not cope with
;	end of year nor with more than a week of data. The older routine, since
;	it had quite different functionality will be retained but not updated.
;	Many part of this routine are derived therefrom.
;
;  History:
;	Prototype: 8/12/92; SJT
;	Add nolabels and minutes keys: 11/12/92; SJT
;	Add noerase key: 29/9/93; SJT
;	Safety when data don't start in requested year: 11/5/94; SJT
;	Fix end offsets in exact range case: 9/11/94; SJT
;	Sort out defined ranges for Delta T > 6 months: 6/1/95; SJT
;	Added date key: 16/2/95; SJT
;	Fix for year >= 2000: 1/12/95; SJT
;-
;   Modified for plot range >10 years: 1/14/05; MID
;
;	Define various working constants.

twenty_years  = 175344.
fifteen_years = 131514.
ten_years = 87660.
five_years = 43830.
three_years = 26280.
two_years = 17520.
one_year = 8760.
six_months = 4392.
four_months = 2976.
three_months = 2232.
two_months = 1488.
one_month = 744.
one_fortnight = 336.
one_week = 168.
one_day = 24.
two_minutes = 1. / 30.
leap_year = one_year + one_day

months = [31,28,31,30,31,30,31,31,30,31,30,31]
cmonths = intarr(13)
cmonths(0) = 0
for i = 1, 12 do cmonths(i) = cmonths(i-1) + months(i-1)
lextra = [intarr(2), intarr(11)+1]

;	Set undefined keys to default values.

if (not keyword_set(nomins)) then nomins = 0
if (not keyword_set(zero)) then zero = 0
if (not keyword_set(exact)) then exact = 0
if (not keyword_set(ytype)) then ytype = 0
if (not keyword_set(yrange)) then yrange = [0., 1.]
if (not keyword_set(yexact)) then yexact = 0
if (not keyword_set(charsize)) then charsize = 1.0
if (not keyword_set(title)) then title = ' '
if (not keyword_set(ytitle)) then ytitle = ' '
if (not keyword_set(xtitle)) then xtitle = ''
if (n_elements(noerase) eq 0) then noerase = 1

if (yrange(0) eq yrange(1)) then yrange=yrange*[.9,1.1]

range = tend - tstart           ; Set time interval.
trange = fltarr(2)

;	Treat the various ranges of time in descending order.
;	After bailing out of extreme values (> 10y or < 2 mins)

minutes = 0
if (range gt twenty_years) then begin	; a bailout for data plots > 20 y.

    hs_err_msg, $
	["**** ERROR **** If you want to plot time ranges greater than", $
	 "                ten years you'll have to modify me to cope."]
    return

endif else if (range lt two_minutes) then begin
    hs_err_msg, ["** WARNING ** If you want to plot time ranges less than", $
		 "              two minutes you'll have to modify me to cope."]
    h0 = fix(tstart)
    trange = ([tstart,tend]-h0) * 60.
    if (keyword_set(day)) then xtitle = timetitle(xtitle, day, hour = $
                                                  h0, date = date)
    plot, noerase=noerase, /nodata, fltarr(2), $
	xrange=trange, xstyle=exact or !x.style, xtitle=xtitle, $
	ytype=ytype, ystyle=yexact or !y.style, ytitle=ytitle, yrange=yrange, $
	title=title, charsize=charsize
    minutes = 1
    return
endif

if (range gt 4. * one_day and not keyword_set(day)) then begin
    hs_err_msg, $
	["**** ERROR **** I can't plot this much data without knowing", $
	 '                the year and day of year of the start...', $
	 '                please pass a keyword DAY with this info']
    return
endif
 
irrflag = 0

;	Firstly consider cases where we will need irregularly spaced ticks,
;	these are the nastiest to deal with. They are the cases where there
;	are more than two months of data to be plotted.
;	In these cases, the year and day MUST be given.
;	The major ticks are irregularly spaced and the minor ticks are drawn
;	explicitly at a specified interval. The year and day are placed on the
;	tick labels.

if (range gt six_months) then begin ; More than 6 months of 
                                    ; data - the messy cases
                                    ; [Why can't a year be an
                                    ; exact no of days?]
    
    sday = norm_time([day, tstart])
    y = date100(sday(0), /print)
    leap = (y mod 4) eq 0
    d = sday(1)
    h0 = hours(day, [sday(0), 1, 0])
    ihy = 24.* (365 + leap)
    
    case 1 of 
	(range gt five_years): begin
	    nguess = fix (range / two_years) + 3	; rough upper limit
							; on number of ticks
	    ngm = 4*(nguess+1)
	    vtick = fltarr(nguess)
	    tnames = strarr(nguess)
	    vminor = fltarr(ngm)

	    if (d eq 1 and tstart eq 0) then begin
		vtick(0) = 0.
		tnames(0) = string(date100(y, /full), format="(I4)")
		zflag = 1
	    endif else begin
		vtick(0) = ihy - h0
		y = date100(y+1, /full)
		tnames(0) = string(y, format="(i4)")
		zflag = 0
	    endelse

	    for j = 1, nguess-1 do begin
		dt = 24. * (730 + ((y mod 4) eq 0) + (((y+1) mod 4) eq 0))
		vtick(j) = vtick(j-1) + dt
		y = date100(y+2, /full)
		tnames(j) = string(y, format="(i4)")
	    endfor
	    vtick = vtick(where(vtick le tend))
	    tnames = tnames(where(vtick le tend))
	    ntick = n_elements(vtick) - 1

	    y = date100(sday(0), /print)
	    h = monthhours([0,6,12],y) - h0
	    vminor = h(where(h ge tstart))
	    while (((ys = max(vminor))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours([6,12],y) + ys
		vminor = [vminor, h]
	    endwhile
	    vminor = vminor(where(vminor le tend))
	    nminor = n_elements(vminor) - 1

	    trange = [tstart, tend]

	    if (xtitle eq '') then xtitle = 'UT Year'

	    irrflag = 1

	end

	(range gt three_years): begin
	    nguess = fix (range / one_year) + 3		; rough upper limit
							; on number of ticks
	    vtick = fltarr(nguess)
	    tnames = strarr(nguess)

	    if (d eq 1 and tstart eq 0) then begin
		vtick(0) = 0.
		tnames(0) = string(date100(y, /full), format="(I4)")
		zflag = 1
	    endif else begin
		vtick(0) = ihy - 24.*(d-1) - tstart
		y = date100(y+1, /full)
		tnames(0) = string(y, format="(i4)")
		zflag = 0
	    endelse

	    for j = 1, nguess-1 do begin
		dt = 24. * (365 + ((y mod 4) eq 0))
		vtick(j) = vtick(j-1) + dt
		y = date100(y+1, /full)
		tnames(j) = string(y, format="(i4)")
	    endfor
	    vtick = vtick(where(vtick le tend))
	    tnames = tnames(where(vtick le tend))
	    ntick = n_elements(vtick) - 1

	    y = date100(sday(0), /print)
	    h = monthhours([0,3,6,9,12],y) - h0
	    vminor = h(where(h ge tstart))
	    while (((ys = max(vminor))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours([3,6,9,12],y) + ys
		vminor = [vminor, h]
	    endwhile
	    vminor = vminor(where(vminor le tend))
	    nminor = n_elements(vminor) - 1

	    if (xtitle eq '') then xtitle = 'UT Year'

	    trange = [tstart, tend]

	    irrflag = 1

	end

	(range gt two_years): begin
	    nguess = fix (range / six_months) + 4	; rough upper limit
							; on number of ticks
	    ngm = 3*(nguess+2)
	    vtick = fltarr(nguess)
	    tnames = strarr(nguess)
	    vminor = fltarr(ngm)

	    h = monthhours([0,6,12],y) - h0
	    vtick = h(where(h ge tstart))
	    while (((ys = max(vtick))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours([6,12],y) + ys
		vtick = [vtick, h]
	    endwhile
	    vtick = vtick(where(vtick le tend))
	    ntick = n_elements(vtick) - 1
	    tnames = strarr(ntick+1)
	    y = date100(sday(0), /print)
	    ily = 365 + ((y mod 4) eq 0)
	    tt =   (vtick+h0) /24. + 1
	    for i = 0, ntick do begin
		if (tt(i) gt ily) then begin
		    tt(i:*) = tt(i:*) - ily
		    y = date100(y+1, /print)
		    ily = 365 + ((y mod 4) eq 0)
                endif
                if (keyword_set(date)) then begin
                    datecon, y, tt(i), mth, dte
                    tnames(i) = strcompress(string(y, mth, dte, format $
                                                   = "(i2.2,'/',a3,'/',i2)"))
                endif else tnames(i) =  $
                  strcompress(string(y, tt(i), format = "(i2.2,'/',i4)"))
	    endfor

	    y = date100(sday(0), /print)
	    h = monthhours([0,2,4,6,8,10,12],y) - h0
	    vminor = h(where(h ge tstart))
	    while (((ys = max(vminor))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours([2,4,6,8,10,12],y) + ys
		vminor = [vminor, h]
	    endwhile
	    vminor = vminor(where(vminor le tend))
	    nminor = n_elements(vminor) - 1

	    if (xtitle eq '') then xtitle = 'UT Date'

	    trange = [tstart, tend]

	    irrflag = 1

	end

	(range gt one_year): begin
	    nguess = fix (range / three_months) + 5	; rough upper limit
							; on number of ticks
	    ngm = 3*(nguess+4)
	    vtick = fltarr(nguess)
	    tnames = strarr(nguess)
	    vminor = fltarr(ngm)

            h = monthhours([0, 3, 6, 9, 12], y) - h0
            locs = where(h ge tstart)
	    vtick = h(locs)
	    while (((ys = max(vtick))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours([3,6,9,12],y) + ys
		vtick = [vtick, h]
	    endwhile
	    vtick = vtick(where(vtick le tend))
	    ntick = n_elements(vtick) - 1
	    tnames = strarr(ntick+1)
	    y = date100(sday(0), /print)
	    ily = 365 + ((y mod 4) eq 0)
	    tt =   (vtick+h0) /24. + 1
	    for i = 0, ntick do begin
		if (tt(i) gt ily) then begin
		    tt(i:*) = tt(i:*) - ily
		    y = date100(y+1, /print)
		    ily = 365 + ((y mod 4) eq 0)
		endif
                if (keyword_set(date)) then begin
                    datecon, y, tt(i), mth, dte
                    tnames(i) = strcompress(string(y, mth, dte, format $
                                                   = "(i2.2,'/',a3,'/',i2)"))
                endif else tnames(i) =  $
                  strcompress(string(y, tt(i), format = "(i2.2,'/',i4)"))
	    endfor

	    y = date100(sday(0), /print)
	    h = monthhours(indgen(13),y) - h0
	    vminor = h(where(h ge tstart))
	    while (((ys = max(vminor))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours(indgen(12)+1,y) + ys
		vminor = [vminor, h]
	    endwhile
	    vminor = vminor(where(vminor le tend))
	    nminor = n_elements(vminor) - 1

	    if (xtitle eq '') then xtitle = 'UT Date'

	    trange = [tstart, tend]

	    irrflag = 1

	end

	(range gt six_months): begin
	    nguess = fix (range / two_months) + 5	; rough upper limit
							; on number of ticks
	    ngm = 2*(nguess+6)
	    vtick = fltarr(nguess)
	    tnames = strarr(nguess)
	    vminor = fltarr(ngm)

            h = monthhours([0, 2, 4, 6, 8, 10, 12], y) - h0
            vtick = h(where(h ge tstart))
            while (((ys = max(vtick))) lt tend) do begin
		y = date100(y+1, /print)
                h = monthhours([2, 4, 6, 8, 10, 12], y) + ys
		vtick = [vtick, h]
	    endwhile
	    vtick = vtick(where(vtick le tend))
	    ntick = n_elements(vtick) - 1
	    tnames = strarr(ntick+1)
	    y = date100(sday(0), /print)
	    ily = 365 + ((y mod 4) eq 0)
            tt = (vtick + h0) /24. + 1
	    for i = 0, ntick do begin
		if (tt(i) gt ily) then begin
		    tt(i:*) = tt(i:*) - ily
		    y = date100(y+1, /print)
		    ily = 365 + ((y mod 4) eq 0)
		endif
                if (keyword_set(date)) then begin
                    datecon, y, tt(i), mth, dte
                    tnames(i) = strcompress(string(y, mth, dte, format $
                                                   = "(i2.2,'/',a3,'/',i2)"))
                endif else tnames(i) =  $
                  strcompress(string(y, tt(i), format = "(i2.2,'/',i4)"))
	    endfor

	    y = date100(sday(0), /print)
	    h = monthhours(indgen(13),y) - h0
	    vminor = h(where(h ge tstart))
	    while (((ys = max(vminor))) lt tend) do begin
		y = date100(y+1, /print)
		h = monthhours(indgen(12)+1,y) + ys
		vminor = [vminor, h]
	    endwhile
	    vminor = vminor(where(vminor le tend))
	    nminor = n_elements(vminor) - 1

	    if (xtitle eq '') then xtitle = 'UT Date'

	    trange = [tstart, tend]

	    irrflag = 1

	end

    endcase
    
endif else if (range ge 96.) then begin		; More than 4 days of
						; data, only the day will go
						; into the label - irrespective
						; of settings.

    case 1 of
	(range gt four_months): begin
	    dt = 30 * 24
	    nminor = 3
	end
	(range gt three_months): begin
	    dt = 30 * 24
	    nminor = 6
	end
	(range gt two_months): begin
	    dt = 15 * 24
	    nminor = 3
	end
        (range gt one_month): begin
	    dt = 10 * 24
	    nminor = 5
        end
        (range gt one_fortnight): begin		; For americans = 2 weeks
	    dt = 5 * 24
	    nminor = 5
        end
        (range gt one_week): begin
	    dt = 2 * 24
	    nminor = 4
        end
        (range ge 96.): begin
	    dt = 24
	    nminor = 4
        end
    endcase

    if (keyword_set(exact)) then begin
	trange(0) = tstart
	if (tstart mod dt) ne 0 then soffset = dt - (tstart mod dt) $
	else			     soffset = 0
	trange(1) = tend
	eoffset = (tend - soffset) mod dt
    endif else begin
	trange(0) = (fix(tstart)/dt)*dt
	if (tstart - trange(0) gt dt/2.) then begin
	    trange(0) = trange(0) + dt/2.0
	    soffset = dt/2.0
	endif else soffset = 0.0

	trange(1) = (fix(tend)/dt)*dt
	if (tend  - trange(1) gt dt/2.0) then begin
	    trange(1) = trange(1) + dt
	    eoffset = 0.
	endif else begin
	    if (trange(1) lt tend) then begin
		trange(1) = trange(1) + dt/2.0
		eoffset = dt/2.0
	    endif else eoffset = 0.
	endelse
    endelse

    ntick = ceil(((trange(1)-eoffset)-(trange(0)+soffset))/dt)
    vtick = dt*findgen(ntick+1) + trange(0) +soffset
;    vtick = vtick(where(vtick le tend))
;    ntick = n_elements(vtick) - 1

    tnames = strarr(ntick+1)
    y = date100(day(0), /print)
    ily = 365 + ((y mod 4) eq 0)
    dd = day(1) + vtick / 24.
    yflag = 0
    omth = 'xxx'
    for i = 0, ntick do begin
	if (dd(i) gt ily) then begin
	    dd(i:*)= dd(i:*) - ily
	    y = date100(y+1, /print)
	    yflag = 1
        endif
        if (keyword_set(date)) then begin
            datecon, y, round(dd(i)), mth, dte
            if (mth ne omth) then tnames(i) = $
              string(mth, dte, format = "(a3,'/',i2)") $
            else tnames(i) = string(dte, format = "(i2)")
            omth = mth
        endif else tnames(i) = string(round(dd(i)), format = "(i3)")
	if (i eq 0 or yflag) then $
		tnames(i) = string(y, format="(I2.2,'/')") + tnames(i)
	yflag = 0
    endfor
    if (xtitle eq '') then xtitle = 'UT Date'
    
endif else if (range gt 3.) then  begin		; More than 3 hours of data.
						; won't need to worry about
						; starting on fractions of an hour
    case 1 of
        (range gt 48.): begin
	    dt = 12
	    nminor = 2
        end
        (range gt 36.): begin
	    dt = 8
	    nminor = 4
        end
        (range gt 24.): begin
	    dt = 6
	    nminor = 6
        end
        (range gt 12.): begin
	    dt = 4
	    nminor = 4
        end
        (range gt 9.): begin
	    dt = 3
	    nminor = 6
        end
        (range gt 6.): begin
	    dt = 2
	    nminor = 4
        end
        (range gt 4.): begin
	    dt = 1
	    nminor = 4
        end
        (range gt 3.): begin
	    dt = 1
	    nminor = 6
        end
    endcase

    if (keyword_set(exact)) then begin
	trange(0) = tstart
	if (tstart mod dt) ne 0 then soffset = dt - (tstart mod dt) $
	else			     soffset = 0
	trange(1) = tend
	eoffset =  (tend - soffset) mod dt
    endif else begin
	trange(0) = (fix(tstart)/dt)*dt
	if (tstart - trange(0) gt dt/2.) then begin
	    trange(0) = trange(0) + dt/2.0
	    soffset = dt/2.0
	endif else soffset = 0.0

	trange(1) = (fix(tend)/dt)*dt
	if (tend  - trange(1) gt dt/2.0) then begin
	    trange(1) = trange(1) + dt
	    eoffset = 0.
	endif else begin
	    if (trange(1) lt tend) then begin
		trange(1) = trange(1) + dt/2.0
		eoffset = dt/2.0
	    endif else eoffset = 0.
	endelse
    endelse

    ntick = ceil(((trange(1)-eoffset)-(trange(0)+soffset))/dt)
    vtick = dt*findgen(ntick+1) + trange(0) +soffset
    vtick = vtick(where(vtick le tend))
    ntick = n_elements(vtick) - 1

    tnames = strarr(ntick+1)
    if (keyword_set(day)) then begin	; Prepare labels with day on.
	y = date100(day(0), /print)
	ily = 365 + ((y mod 4) eq 0)
	h0 = fix(vtick)
	h = h0 mod 24
	dd = day(1) + h0/24
	locs = where(h lt 0)
	if (locs(0) ne -1) then h(locs) = h(locs) + 24
	if (max(dd) lt ily) then yflag = -1 $
        else			 yflag = 1
        omth = 'xxx'
	for i = 0, ntick do begin
	    if (dd(i) gt ily and yflag ne -1) then begin
		dd(i:*) = dd(i:*) - ily
		y = date100(y+1, /print)
		yflag = 1
	    endif
	    if ((not keyword_set(zero)) or (i eq 0) or $
		    (h(i) eq 0 and i ne ntick)) then begin
                if (keyword_set(nomins)) then begin
                    if (keyword_set(date)) then begin
                        datecon, y, round(dd(i)), mth, dte
                        if (mth ne omth) then tnames(i) =  $
                          string(mth, dte, round(h(i)), $
                                 format = "(a3,'/',i2,1x,i2.2)") $
                        else tnames(i) = string(dte, round(h(i)), $
                                                format = $
                                                "(i2,1x,i2.2)")
                        omth = mth
                    endif else  tnames(i) = string(round([dd(i), h(i)]), $
                                                   format = "(i3,1x,i2.2)")
		endif else begin
                    if (keyword_set(date)) then begin
                        datecon, y, round(dd(i)), mth, dte
                        if (mth ne omth) then tnames(i) =  $
                          string(mth, dte, round(h(i)), $
                                 format = "(a3,'/',i2,1x,i2.2,':00')") $
                        else tnames(i) =  $
                          string(dte, round(h(i)), $
                                 format = "(i2,1x,i2.2,':00')")
                        omth = mth
                    endif else  tnames(i) =  $
                      string(round([dd(i), h(i)]), $
                             format = "(i3,1x,i2.2,':00')")
		endelse
	    endif else begin
		if (keyword_set(nomins)) then begin
		    tnames(i) = string(h(i)+0.5, format = "(I2.2)")
		endif else begin
		    tnames(i) = string(h(i)+0.5, format = "(I2.2,':00')")
		endelse
	    endelse
	    if (yflag eq 1) then begin
		tnames(i) = string(y, format="(I2.2,'/')") + tnames(i)
		yflag = 0
	    endif
	endfor
        if (yflag eq -1) then xtitle = timetitle(xtitle, day(0), $
                                                 date = date) $ 
	else if (xtitle eq '') then xtitle = 'UT Date'
    endif else begin
	h0 = fix(vtick)
	h = h0 mod 24
	locs = where(h lt 0)
	if (locs(0) ne -1) then h(locs) = h(locs)+24
	for i = 0, ntick do begin
	    if (keyword_set(nomins)) then begin
	        tnames(i) = string(h(i)+0.5, format = "(I2.2)")
	    endif else begin
	        tnames(i) = string(h(i)+0.5, format = "(I2.2,':00')")
	    endelse
	endfor
    endelse
endif else begin		; Less than 3 hours of data, here we do have
				; to worry about non integral hours at the
				; start and end of the plot, work round it
				; by working with a range in minutes.

    rmins = range*60.

    case 1 of
	(rmins gt 120.): begin
	    dt = 30
	    nminor = 6
	end
	(rmins gt 60.): begin
	    dt = 15
	    nminor = 3
	end
	(rmins gt 30.): begin
	    dt = 10
	    nminor = 5
	end
	(rmins gt 15.): begin
	    dt = 5
	    nminor = 5
	end
	(rmins gt 7.): begin
	    dt = 2
	    nminor = 2
	end
	(rmins gt 4.): begin
	    dt = 1
	    nminor = 4
	end
	(rmins gt 2.): begin
	    dt = 1
	    nminor = 6
	end
    endcase

    mstart = tstart*60.
    mend   = tend*60.

    if (keyword_set(exact)) then begin
	trange(0) = mstart
	if (mstart mod dt) ne 0 then soffset = dt - (mstart mod dt) $
	else			     soffset = 0
	trange(1) = mend
        eoffset = (mend - soffset) mod dt
    endif else begin
	trange(0) = (fix(mstart)/dt)*dt
	if (mstart - trange(0) gt dt/2.) then begin
	    trange(0) = trange(0) + dt/2.0
	    soffset = dt/2.0
	endif else soffset = 0.0

	trange(1) = (fix(mend)/dt)*dt
	if (mend - trange(1) gt dt/2.0) then begin
	    trange(1) = trange(1) + dt
	    eoffset = 0.
	endif else begin
	    if (trange(1) lt mend) then begin
	        trange(1) = trange(1) + dt/2.0
		eoffset = dt/2.0
	    endif else eoffset = 0.
	endelse
    endelse

    ntick = ceil(((trange(1)-eoffset)-(trange(0)+soffset))/dt)
    vtick = dt*findgen(ntick+1) + trange(0) +soffset
    vtick = vtick(where(vtick le mend))
    ntick = n_elements(vtick) - 1

    trange = trange/60.0
    vtick = vtick/60.0

    tnames = strarr(ntick+1)

    if (keyword_set(day)) then begin	; Prepare labels with day on.
	y = date100(day(0), /print)
	ily = 365 + ((y mod 4) eq 0)
	h0 = fix(vtick)
	m = (vtick-h0)*60.
	h = h0 mod 24
	dd = day(1) + h0/24
	locs =  where(h lt 0)
	if (locs(0) ne -1) then h(locs) = h(locs)+24
	if (max(dd) lt ily) then yflag = -1 $
        else			 yflag = 1
        omth = 'xxx'
	for i = 0, ntick do begin
	    if (dd(i) gt ily) then begin
		dd(i:*) = dd(i:*) - ily
		y = date100(y+1, /print)
		ily = 365 + ((y mod 4) eq 0)
		yflag = 1
	    endif
	    if ((not keyword_set(zero)) or (i eq 0) or $
			(h(i) eq 0 and m(i) eq 0 and i ne ntick)) then begin
                if (keyword_set(date)) then begin
                    datecon, y, round(dd(i)), mth, dte
                    if (mth ne omth) then tnames(i) =  $
                      string(mth, dte, round([h(i), m(i)]), $
                             format = "(a3,'/',i2,1x,i2.2,':',I2.2)") $
                    else tnames(i) = string(dte, round([h(i), m(i)]), $
                                            format = "(i2,1x,i2.2,':',I2.2)")
                    omth = mth
                endif else  tnames(i) = string(round([dd(i), h(i), m(i)]), $
                                               format =  $
                                               "(i3,1x,i2.2,':',I2.2)")
	    endif else begin
                tnames(i) = string(round([h(i), m(i)]), format = $
                                   "(I2.2,':',i2.2)")
	    endelse
	    if (yflag eq 1) then begin
		tnames(i) = string(y, format="(I2.2,'/')") + tnames(i)
		yflag = 0
	    endif
	endfor
        if (yflag eq -1) then xtitle = timetitle(xtitle, day(0), $
                                                 date = date) $
	else if (xtitle eq '') then xtitle = 'UT Date'
    endif else begin
	h0 = fix(vtick)
	m = (vtick-h0)*60.
	h = h0 mod 24
	locs = where(h lt 0)
	if (locs(0) ne -1) then h(locs) = h(locs)+24
	for i = 0, ntick do $
	    tnames(i) = string([h(i),m(i)]+0.5, format = "(I2.2,':',i2.2)")
    endelse
endelse

if (keyword_set(nolabels)) then begin
    tnames(*) = ' '
    xtitle = ''
endif

if (irrflag) then begin
    plot, /nodata, noerase=noerase, fltarr(2), charsize=charsize, $
	title=title, xtitle=xtitle, ytitle=ytitle, $
	ystyle=yexact or !y.style, ytype=ytype, yrange=yrange, $
	xstyle=1 or !x.style, xticks=ntick, xtickv=vtick, xtickname=tnames, $
	xrange = trange

    axis, xax=0, xticks=nminor, xtickv=vminor, $
	xtickname=strarr(nminor+1)+' ', xticklen=!p.ticklen/2.
    axis, xax=1, xticks=nminor, xtickv=vminor, $
	xtickname=strarr(nminor+1)+' ', xticklen=!p.ticklen/2.
endif else begin
    plot, /nodata, noerase=noerase, fltarr(2), charsize=charsize, $
      title = title, xtitle = xtitle, ytitle = ytitle, $
      ystyle = yexact or !Y.style, ytype = ytype, yrange = yrange, $
      xstyle = 1 or !X.style, xticks = ntick, xtickv = vtick, $
      xtickname = tnames, xrange = trange, xminor = nminor
endelse

end
